#!/usr/bin/env python -*- coding: utf-8 -*-
#
# Biblioteka zawiera klasy do sterowania generatorem i oscyloskopem
# w pracowni.
#
# Paweł Klimczewski, 14 listopada 2024

import subprocess, sys, time, math

#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
try:
    get_ipython()
    INTERACTIVE_PYTHON = True
except NameError:
    INTERACTIVE_PYTHON = False
#----------------------------------------------------------------------------
#
# UZUPEŁNIENIE BRAKUJĄCEGO OPROGRAMOWANIA
#
# Sterowanie przyrządami wymaga dodatkowej biblioteki
#
# https://pyvisa.readthedocs.io/en/latest/
#
# nieobecnej w Pythonie Anaconda. Biblioteka visa zapewnia komunikację
# z urządzeniami pomiarowymi połączonymi z komputerem. Biblioteka może
# korzystać m.in. z darmowych sterowników LabVIEW
#
# https://pyvisa.readthedocs.io/en/latest/introduction/getting.html#backend
#
# Biblioteka zostanie automatycznie zainstalowana w momencie uruchomienia
# programu korzystającego z biblioteki tik.
#
try:
    import pyvisa as visa
    from termcolor import colored
except ImportError:
    # Automatyczna instalacja brakującego oprogramowania.
    print("Installing packages. Please wait.")
    subprocess.check_call([sys.executable, "-m", "pip", "install",
        "--upgrade", "--user", "--no-cache-dir", "-q", "-q",
        "pip", "termcolor", "pyvisa", "pyvisa-py", "pyusb", "pyserial",
        "zeroconf"])
    if INTERACTIVE_PYTHON:
        # Jeżeli importujemy bibliotekę w środowisku interaktywnym, np.
        # w notesie, to sami musimy ponownie uruchomić jądro pythona.
        print("Restart kernel!")
    else:
        # Jeżeli importujemy bibliotekę w trybie programu, to przerywamy jego
        # pracę.
        print("Run program again!")
        sys.exit(0)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# KLASY REPREZENTUJĄCE URZĄDZENIA POMIAROWE
#
# Rozpoczynamy tworzenie drugiej grupy klas reprezentujących sterowania
# urządzeniami. Rozwiązania zapiszemy w postaci ciągu klas InstrumentBase0,
# InstrumentBase1 itd. Kolejne klasy są bogatsze o nową funkcjonalność. Jest
# to ilustracja cechy programowania obiektowego jaką jest dziedziczenie klas.
#
# Rozwiązanie zagadnień realizowanych przez klasę InstrumentBase5 wykorzystuje
# dwie funkcje składowe tej klasy check_event_queue,
# send_and_wait_until_finished. Działanie tych funkcji jest związane
# z konkrentnym przyrządem i zostanie określone we właściwej klasie pochodnej.
# Jest to ilustracja kolejnej cechy programowania obiektowego - polimorfizmu.
#
class InstrumentBase0:
    # Konstruktor nawiązuje połączenie z przyrządem.
    def __init__(self):
        rm = visa.ResourceManager()
        l = rm.list_resources()
        if len(l)==0:
            raise Exception('No devices found!')
        if not INTERACTIVE_PYTHON and len(sys.argv)>1:
            # Jeżeli uruchamiając program podamy po nazwie programu dodatkowe
            # wartości liczbowe to oznaczają one wybory kolejnych przyrządów
            # z listy.
            k = int(sys.argv[1])
            sys.argv.remove(sys.argv[1])
        else:
            # Wypisanie listy przyrządów podłączonych do komputera.
            n = 0
            for id in l:
                n += 1
                print(f'{n}) {id}')
            # Wybranie przyrządu przez użytkownika.
            while True:
                try:
                    k = int(input(f'Select {self.__class__.__name__} '))
                    if k>=1 and k<=n:
                        in_use = False
                        for u in rm.list_opened_resources():
                            if u.resource_name == l[k-1]:
                                in_use = True
                                print('Resource in use!')
                                break
                        if not in_use:
                            break
                except ValueError:
                    continue
        self._s = rm.open_resource(l[k-1])
        self._s.timeout = 10000 # 10000 ms = 10 s
        # Dla oscyloskopu Tektronix z łączem RS232.
        self._s.baud_rate = 19200
        self._s.flow_control = visa.constants.ControlFlow.rts_cts
    #------------------------------------------------------------------------
    def __del__(self):
        print(f'Deleting object {self.__class__.__name__}.'
              ' Closing connection with device.')
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# AUTOMATYCZNE ZWALNIANIE ZASOBÓW
#
# Programy komputerowe korzystają ze zmiennych, plików, połączeń. Wszystkie te
# elementy nazywamy zasobami. Dobrym zwyczajem jest zwalnianie zasobów, kiedy
# nie są już dłużej potrzebne. Rozpoczynamy od automatycznego zamykania
# połączenia z przyrządami.
#
# Środowisko języka Python samo dba o automatyczne zwalnianie zasobów.
# Możliwości programisty są w tym względzie ograniczone. Jedną z możliwości
# jest skorzystanie z instrukcji with. Wymaga to właściwego przygotowania
# obiektów będących argumentami instrukcji with. Obiekty utworzone
# w kontekście instrukcji with wykonują metody enter i exit, odpowiednio na
# początku i na końcu bloku instrukcji związanych z instrukcją with.
#
class InstrumentBase1(InstrumentBase0):
    def __init__(self):
        super().__init__()
        self._depth = 0 # Licznik zagnieźdzeń instrukcji with.
    #------------------------------------------------------------------------
    def __enter__(self):
        self._depth += 1
        return self
    #------------------------------------------------------------------------
    def __exit__(self, *args):
        if self._depth>0:
            self._depth -= 1
            if self._depth==0:
                self._s.close()
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# KOMUNIKACJA Z UŻYTKOWNIKIEM
#
# Trudno jest napisać prawidłowy program za pierwszym razem. Często
# niedoskonałości programu zauważamy dopiero po jego uruchomieniu.
#
# Mechanizm wypisywania ułatwia diagnozowanie pracy programu. Za pomocą
# argumentu verbose użytkownik może wybrać trzy poziomy komunikacji.
#
# 1. (verbose=0) Biblioteka nie wypisuje żadnych informacji podczas
#    komunikacji z przyrządami.
#
# 2. (verbose=1) Wypisywane są przesyłane rozkazy oraz odczytane odpowiedzi.
#
# 3. (verbose=2) Jak wyżej z uzupełnieniem o:
#
#    - informacjach o pauzie przed wysłaniem rozkazu,
#
#    - informacjach o czasie odbioru odpowiedzi.
#
# Funkcja print umożliwia wypisywanie komunikatów ze sprawdzeniem
# obowiązującego poziomu komunikacji i w wybranym kolorze.
#
class InstrumentBase2(InstrumentBase1):
    def __init__(self):
        super().__init__()
        self.verbose()
    #------------------------------------------------------------------------
    def quiet(self):
        self._verbose = 0
    #------------------------------------------------------------------------
    def verbose(self):
        self._verbose = 1
    #------------------------------------------------------------------------
    def debug(self):
        self._verbose = 2
    #------------------------------------------------------------------------
    def _print(self, level, message='', color=''):
        if self._verbose>=level:
            print(message if len(color)==0 else colored(message, color),
                end='\n' if len(message)==0 else ' ', flush=True)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# NIEDOSKONAŁOŚCI KOMUNIKACJI Z PRZYRZĄDAMI
#
# Oprogramowanie przyrządów Rigol zawiera błędy. Zbyt szybka komunikacja
# prowadzi do „zawieszania się” przyrządów.
#
# Komunikacja z przyrządami polega na wysyłaniu rozkazów (poleceń lub zapytań)
# i - w przypadku zapytań - odczytywaniu odpowiedzi. Doświadczenie podpowiada,
# że w pewnych sytuacjach nie możemy przesyłać rozkazów zbyt szybko po sobie
# ani zbyt szybko odczytywać odpowiedzi po przesłanym zapytaniu. Wprowadzamy
# dwie stałe.
#
# - _wr_delay - minimalny odstęp czasowy pomiędzy wysłaniem zapytania
#   a rozpoczęciem odczytywania odpowiedzi.
#
# - _ww_delay - minimalny odstęp czasowy pomiędzy przesłaniem dwóch kolejnych
#   rozkazów.
#
# Na podstawie obserwacji zostały dobrane następujące wartości.
#
# | urządzenie           | interfejs | wr_delay [s] | ww_delay [s] |
# | -------------------- | --------- | ------------ | ------------ |
# | generator Rigol      | VISA      | 0,25         | 0,25         |
# | oscyloskop Rigol     | LXI       | -            | 0,2          |
# | oscyloskop Rigol     | VISA      | -            | -            |
# | oscyloskop Tektronix | RS232     | -            | -            |
#
# Definiujemy funkcje write i read, których zadaniem będzie przesyłanie
# rozkazów i odczytywanie odpowiedzi z uwzględnieniem odstępów czasowych.
#
# Dodatkowo dla verbose=1 funkcje będą wypisywać treści przesyłanych rozkazów
# i odczytywanych odpowiedzi. (Treści poleceń będą wypisywane w kolorze
# zółtym.) Dla verbose=2 będą wypisywane informacje o wprowadzanych pauzach
# czasowych oraz czasie potrzebnym do odczytania odpowiedzi związanej
# z zapytaniem.
#
class Timer:
    def __init__(self, obj, op):
        self._obj = obj
        self._op = op
    #------------------------------------------------------------------------
    def __enter__(self):
        self._obj._sleep()
        self._t0 = time.time()
    #------------------------------------------------------------------------
    def __exit__(self, *args):
        self._obj._last_rdwr_t = time.time()
        self._obj._print(2,
            f'[{self._op}={int(1000*(self._obj._last_rdwr_t-self._t0))} ms]',
                'cyan')
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
class InstrumentBase3(InstrumentBase2):
    def __init__(self):
        super().__init__()
        self._delay = 0
        self._last_rdwr_t = 0
    #------------------------------------------------------------------------
    def _sleep(self):
        dt = self._delay - (time.time() - self._last_rdwr_t)
        if dt>0:
            time.sleep(dt)
            self._print(2, '[wait=' f'{1000*dt:.0f}' ' ms]', 'cyan')
    #------------------------------------------------------------------------
    def _write(self, c, verbose=1):
        p = c.partition(';')
        self._print(verbose, p[0], '' if '?' in p[0] else 'yellow')
        tail = ''.join(p[1:])
        if len(tail)>0:
            self._print(2, tail)
        with Timer(self, 'wr'):
            self._s.write(c)
    #------------------------------------------------------------------------
    def _read(self):
        with Timer(self, 'rd'):
            b = self._s.read_raw()
        if len(b)==0:
            raise Exception('no reply from instrument')
        return b
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# AUTOMATYCZNE ROZPOZNAWANIE ODPOWIEDZI BINARNYCH
#
# Rozmiar odpowiedzi to 1152066 bajtów. Odpowiedź binarna składa się
# z nagłówka i ciągu bajtów reprezentujących obraz. Rolą nagłówka jest
# zasygnalizowanie faktu wystąpienia odpowiedzi binarnej i przekazanie
# rozmiaru danych.
#
class InvalidData(Exception):
    pass
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
class InstrumentBase4(InstrumentBase3):
    def _read(self, verbose=1, check=False):
        b = super()._read()
        if b[0:1].decode('ascii')=='#':
            # Odpowiedź binarna.
            while len(b)<2:
                b += super()._read()
            if not b[1:2].decode('ascii').isdigit():
                raise Exception('get: invalid header length')
            n1 = int(b[1:2].decode('ascii'))
            while len(b)<2+n1:
                b += super()._read()
            if not b[2:2+n1].decode('ascii').isdigit():
                raise Exception('get: invalid data length')
            n2 = int(b[2:2+n1].decode('ascii'))
            while len(b)<2+n1+n2+1:
                b += super()._read()
            while b[-1]!=b'\n'[0]:
                b += super()._read()
            self._print(verbose, f'--> {len(b)} byte(s)')
            if check and n2==0:
                raise InvalidData('empty binary answer')
            return list(b[2+n1:2+n1+n2])
        else:
            # Odpowiedź tekstowa.
            while b[-1]!=b'\n'[0] and b.decode('ascii')!='command error':
                b += super()._read()
            s = b.decode('iso-8859-1').rstrip()
            try:
                s = int(s)
                if s == 0xffffffff:
                    s = math.nan
            except:
                try:
                    s = float(s)
                    if s == 9.9e+37:
                        s = math.nan
                except:
                    pass
            self._print(verbose, f'--> {s}')
            if check and math.isnan(s):
                raise InvalidData('invalid answer')
            return s
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# FUNKCJA TALK
#
# - Funkcja posiada jeden argument - napis lub listę napisów. Każdy napis
#   zawiera polecenie lub zapytanie. Niech m oznacza liczbę poleceń, a n -
#   liczbę zapytań.
#
# - Rezultat funkcji zawiera n odpowiedzi odpowiadających n zapytaniom.
#   Pojedyncza odpowiedź jest zwracana jako wartość, większa liczba odpowiedzi
#   - jako lista wartości.
#
# - Po przesłaniu każdego polecenia czekamy na zakończenie jego wykonania -
#   funkcja send_and_wait_until_finished.
#
# - Po przesłaniu każdego rozkazu sprawdzamy dziennik błędów - funkcja
#   check_event_queue.
#
# Sposób czekania na zakończenie wykonania polecenia zależy od konkretnego
# urządzenia i zostanie określony we właściwej klasie pochodnej przez ponowne
# zapisanie funkcji send_and_wait_until_finished. Tak samo sposób sprawdzenia
# dziennika zostanie określony we właściwej klasie pochodnej przez ponowne
# zapisanie funkcji check_event_queue.
#
# Funkcje send_and_wait_until_finished i check_event_queue zapisane w klasie
# InstrumentBase4 nie realizują właściwych sobie czynności. Są jedynie
# zapowiedzią, że właściwe funkcje zostaną określone w klasach pochodnych.
# W programowaniu obiektowym funkcje takie nazywamy abstrakcyjnymi.
#
# Mechanizm, dzięki któremu funkcja talk zapisana w klasie InstrumentBase4
# będzie korzystała z funkcji send_and_wait_until_finished oraz
# check_event_queue zapisanych w klasach pochodnych to polimorfizm.
#
class InstrumentBase5(InstrumentBase4):
    def __init__(self):
        super().__init__()
        self.RIGOL = 'Rigol' in self.__class__.__name__
    #------------------------------------------------------------------------
    # Właściwa implementacja w klasie reprezentującej konkretne urządzenie.
    def _check_event_queue(self, cmd):
        pass
    #------------------------------------------------------------------------
    # Właściwa implementacja w klasie reprezentującej konkretne urządzenie.
    def _send_and_wait_until_finished(self, cmd):
        self._write(cmd)
    #------------------------------------------------------------------------
    def talk(self, cmds, check=False):
        # Jeżeli argument cmd jest napisem to zmienimy go w jednoelementową
        # tablicę.
        if isinstance(cmds, str):
            cmds = [cmds]
        ret_val = []
        for c in cmds:
            try:
                c = c.strip()
                # Podział rozkazu złożonego na pojedyncze rozkazy.
                p = [e.strip() for e in c.split(';')]
                # Część rozkazu przed pierwszą spacją.
                p = [e.partition(' ')[0] for e in p]
                # Identyfikuję rozkazy będące zapytaniami.
                p = [e.endswith('?') for e in p]
                if True in p:
                    # Przynajmniej jedno zapytanie.
                    self._write(c)
                    ret_val.append(self._read(1, check))
                else:
                    # Same polecenia.
                    self._send_and_wait_until_finished(c)
                # Sprawdzenie dziennika błędów.
                self._check_event_queue(c)
            finally:
                self._print(1)
        # Większą ilość odpowiedzi przekazuję jako tablicę.
        # Pojedynczą odpowiedź przekazuję jako wartość.
        return ret_val if len(ret_val)>1 else ret_val[0] if len(ret_val)>0 \
            else None
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# ODCZYTANIE DZIENNIKA ZDARZEŃ W PRZYRZĄDACH RIGOL
#
# Przyrządy Rigol charakteryzują się jednolitym sposobem dostępu do dziennika
# zdarzeń.
#
class RigolEventQueue(InstrumentBase5):
    #------------------------------------------------------------------------
    # Generator i oscyloskop Rigol posługują się identycznym sposobem
    # odczytywania dziennika zdarzeń.
    def _check_event_queue(self, cmd):
        self._print(2)
        self._write('system:error?', 2)
        err = self._read(2)
        # Obliczam wartość przed przecinkiem.
        if int(err.split(',')[0])!=0:
            raise Exception(f'{cmd} -> {err}')
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# KOŃCOWA KLASA REPREZENTUJĄCA GENERATOR RIGOL
#
# Generator komunikuję się przez magistralę USB. Konieczne są dodatkowe pauzy
# czasowe.
#
class RigolGenerator(RigolEventQueue):
    def __init__(self):
        super().__init__()
        self._delay = 0.25
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# AUTOMATYCZNE DOPASOWANIE JEDNOSTKI NA OSI OY OSCYLOSKOPU
#
# Cechą wspólną oscyloskopów Rigol i Tektronix jest 8-mio bitowy przetwornik
# A/C. Pojedyncza podziałka na osi OY odpowiada 25 jednostkom przetwornika.
# Możemy zapisać jedną funkcję automatycznego dopasowania jednostki na osi OY.
#
class AutoScale:
    # Właściwa implementacja w klasie reprezentującej konkretne urządzenie.
    def _check_scale_offset(self, scale, offset):
        return scale, offset
    #------------------------------------------------------------------------
    def _set_scale(self, channel, new_scale, scale):
        new_scale, _ = self._check_scale_offset(new_scale, 0)
        if new_scale!=scale:
            prefix = f'channel{channel}' if self.RIGOL else f'ch{channel}'
            if self.talk([
                f'{prefix}:scale {new_scale:.2e}',
                f'{prefix}:scale?'])!=scale:
                return True
        return False
    #------------------------------------------------------------------------
    def _set_offset(self, channel, new_offset, scale):
        _, new_offset = self._check_scale_offset(scale, new_offset)
        if self.RIGOL:
            self.talk(f'channel{channel}:offset {new_offset:.2e}')
        else:
            self.talk(f'ch{channel}:position {new_offset/scale:.2e}')
    #------------------------------------------------------------------------
    # Oscyloskopy Rigol i Tektronix posiadają 8-mio bitowe przetworniki A/C.
    # Stąd można zaproponować taki sam algorytm automatycznego dopasowania
    # jednostki i przesunięcia dla osi OY.
    #
    # Ekran oscyloskopu wyświetla obraz odpowiadający 8 jednostkom na osi OY.
    # Jednostce OY odpowiada zmiana wartości o 25, a całemu ekranowi - o 200.
    # Oś OX znajduje się w połowie wysokości ekranu.
    #
    # Jeżeli wykres zajmuje mniej niż 90% ekranu to zwiększam czułość toru
    # sygnałowego. Jeżeli wykres nie mieści się na ekranie to zmniejszam
    # czułość toru sygnałowego o 20%.
    def _autoscale(self, channels, v):
        if type(channels)!=list:
            channels=[channels]
        ret_val = False
        for i in channels:
            c = v[f'ch{i:d}']
            best_scale = max(abs(max(c['u'])),abs(min(c['u'])))/4
            # Unikam zbyt gwałtownego wzrostu czułości.
            if best_scale/c['scale']<0.125:
                best_scale = c['scale']*0.125
            new_scale = None
            if c['quality']==0:
                new_scale = best_scale*1.2
            else:
                ratio = best_scale/c['scale']
                if ratio<0.9 or ratio>1.1:
                    new_scale = best_scale
            if new_scale:
                b = self._set_scale(i, new_scale, c['scale'])
                ret_val = ret_val or b
            if c['offset']!=0:
                self._set_offset(i, 0, 1)
                ret_val = True
        return ret_val
    #------------------------------------------------------------------------
    def _best_view(self, v):
        u = v['u']
        u_min = min(u)
        u_max = max(u)
        u_avg = (u_min+u_max)/2
        # Idealne dopasowanie.
        scale0, offset0 = (u_max-u_min)/8, -u_avg
        # Unikam zbyt gwałtownego wzrostu czułości.
        if scale0/v['scale']<0.125:
            scale0 = v['scale']*0.125
        # Sprawdzam możliwości oscyloskopu.
        scale1, offset1 = self._check_scale_offset(scale0, offset0)
        if offset1==offset0:
            return scale1, offset1
        # Tak dobieram scale2 aby ta strona wykresu, która bardziej wystaje
        # dotykała brzegu ekranu: u_max+offset = 4*scale lub u_min+offset =
        # -4*scale
        scale2 = (u_max+offset1)/4 if u_avg>=0 else -(u_min+offset1)/4
        # Nie potrzeba sprawdzać scale2. Będzie dobre skoro u_min i u_max są
        # wyliczone na podstawie zmierzonego u.
        VAL = 0.5 if self.RIGOL else 0.2
        if scale1<=VAL and scale2<=VAL or scale1>VAL:
            return scale2, offset1
        # Dla scale>VAL mamy większy zakres offsetu.
        offset2 = -u_avg
        _, offset3 = self._check_scale_offset(VAL, offset2)
        if offset2==offset3:
            return VAL, offset3
        # Liczę scale3 dla offset3.
        scale3 = (u_max+offset3)/4 if u_avg>=0 else -(u_min+offset3)/4
        return scale3, offset3
    #------------------------------------------------------------------------
    def _autoscale_autooffset(self, channels, v):
        if type(channels)!=list:
            channels=[channels]
        ret_val = False
        for i in channels:
            key = f'ch{i:d}'
            c = v[key]
            b = c['bytes']
            b_min = min(b)
            b_max = max(b)
            scale_max, _ = self._check_scale_offset(1e6, 0)
            if c['scale']==scale_max:
                if b_min==0 and b_max==255 or c['quality']>=90:
                    continue
                y_incr = c['yincrement'] if self.RIGOL else c['ymult']
                if b_min==0:
                    new_scale, new_offset = \
                        c['scale'], c['offset'] + (255-b_max)*y_incr
                elif b_max==255:
                    new_scale, new_offset = \
                        c['scale'], c['offset'] - b_min*y_incr
                else:
                    new_scale, new_offset = self._best_view(c)
            else:
                if c['quality']==0:
                    new_scale, new_offset = 1.2*c['scale'], c['offset']
                elif c['quality']>=90:
                    continue
                else:
                    new_scale, new_offset = self._best_view(c)
            new_scale, new_offset = self._check_scale_offset(new_scale,
                new_offset)
            scale_diff = self._set_scale(i, new_scale, c['scale'])
            self._set_offset(i, new_offset, c['scale'])
            ret_val = ret_val or scale_diff
        return ret_val
    #------------------------------------------------------------------------
    def autoscale(self, channels, v, autooffset=True):
        return self._autoscale_autooffset(channels, v) if autooffset else \
            self._autoscale(channels, v)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
class Slower:
    def __init__(self, obj):
        self._obj = obj
    #------------------------------------------------------------------------
    def __enter__(self):
        self._obj._delay = 0.25
    #------------------------------------------------------------------------
    def __exit__(self, *args):
        self._obj._delay = 0
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# KLASA REPREZENTUJĄCA OSCYLOSKOP RIGOL
#
# W klasie określamy.
#
# 1. Sposób czekania na zakończenie wykonywania polecenia.
#
# 2. Sposób wykonania pojedynczego pomiaru.
#
# 3. Sposób odczytanie wyników pomiaru dla określonego toru sygnałowego.
#
class RigolOscilloscope(RigolEventQueue, AutoScale):
    # Liczba podziałek na osi OX.
    NDIVX = 12
    #------------------------------------------------------------------------
    def __init__(self):
        super().__init__()
        self.slower = Slower(self)
    #------------------------------------------------------------------------
    def _send_and_wait_until_finished(self, cmd):
        # TODO
        # Jedną z niedoskonałości oscyloskopów Rigol jest zawieszanie się
        # komunikacji podczas zapytania *opc?. Wysłanie po sobie dwóch
        # rozkazów *rst i *opc? lub *cls i *opc? powoduje zawieszenie się
        # połączenia USB ale działa prawidłowo przy połączeniu LXI. Wysłanie
        # polecenia złożonego *rst; *opc? lub *cls; *opc? powoduje zawieszenie
        # się połączenia LXI ale działa prawidłowo przy połączeniu USB.
        # Oscyloskopy w pracowniach komunikują się przez USB.

        # self._write(cmd + '; *opc?')
        # self._read(2)

        self._write(cmd)
        self._print(2)
        self._write('*opc?', 2)
        self._read(2)
    #------------------------------------------------------------------------
    def _check_scale_offset(self, scale, offset):
        scale = min(max(scale, 0.001), 10)
        if scale<0.5:
            offset = min(max(offset, -2), 2)
        else:
            offset = min(max(offset, -100), 100)
        return scale, offset
    #------------------------------------------------------------------------
    def read_waveforms(self, channels):
        if type(channels)!=list:
            channels = [channels]
        for j in range(10):
            v = {}
            v['time'] = time.asctime()
            v['time_scale'] = self.talk('timebase:main:scale?', True)
            for i in channels:
                c = {}
                v[f'ch{i}'] = c
                (c['bytes'], c['xincrement'], c['yincrement'],
                c['yreference'], c['yorigin'], c['scale'],
                c['offset']) = \
                self.talk([
                    f'waveform:source channel{i}',
                    'waveform:data?',
                    'waveform:xincrement?',
                    'waveform:yincrement?',
                    'waveform:yreference?',
                    'waveform:yorigin?',
                    f'channel{i}:scale?',
                    f'channel{i}:offset?'], True)
                b_min = min(c['bytes'])
                b_max = max(c['bytes'])
                c['quality'] = 0 if b_min == 0 or b_max == 255 \
                    else (b_max-b_min)/2
                c['u'] = [(b-c['yorigin']-c['yreference'])*c['yincrement']
                    for b in c['bytes']]
                div = c['scale']
                c['u_err'] = [(0.03 if div>=0.01 else 0.04)*abs(u)+0.1*div+\
                    0.002+0.1*abs(c['offset']) for u in c['u']]
            break
        return v
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#
# KLASA REPREZENTUJĄCA OSCYLOSKOP TEKTRONIX
#
# W klasie definiujemy.
#
# 1. Sposób odczytu dziennika zdarzeń.
#
# 2. Sposób oczekiwania na zakończenie wykonywania wcześniejszych poleceń.
#
# 3. Sposób wykonania pojedynczego pomiaru.
#
# 4. Sposób odczytania wyników pomiaru z określonego toru sygnałowego.
#
class TektronixOscilloscope(InstrumentBase5, AutoScale):
    # Liczba podziałek na osi OX.
    NDIVX = 10
    #------------------------------------------------------------------------
    # Dostęp do dziennika zdarzeń.
    def _check_event_queue(self, cmd):
        self._print(2)
        self._write('*esr?', 2)
        errcode = int(self._read(2))
        if errcode==0:
            return
        while True:
            self._print(2)
            self._write('evmsg?', 2)
            err = self._read(2)
            # Obliczam wartość przed przecinkiem.
            parts = err.partition(',')
            if int(parts[0])==0:
                break
            msg = parts[2].lower()
            if 'warning' in msg or 'power on' in msg:
                continue
            raise Exception(f'{cmd} -> {err}')
    #------------------------------------------------------------------------
    # Czekanie na zakończenie wykonania polecenia.
    def _send_and_wait_until_finished(self, cmd):
        self._write(cmd)
        while True:
            self._print(2)
            self._write('busy?', 2)
            r = self._read(2)
            if r==':BUSY 0' or int(r)==0:
                break
            time.sleep(0.1)
    #------------------------------------------------------------------------
    def _check_scale_offset(self, scale, offset):
        scale = min(max(scale, 0.002), 5)
        if scale<=0.2:
            offset = min(max(offset, -2), 2)
        else:
            offset = min(max(offset, -50), 50)
        return scale, offset
    #------------------------------------------------------------------------
    def read_waveforms(self, channels):
        if type(channels)!=list:
            channels = [channels]
        v = {}
        v['time'] = time.asctime()
        v['time_scale'] = self.talk('horizontal:main:scale?')
        for i in channels:
            c = {}
            v[f'ch{i}'] = c
            (c['bytes'], c['xincr'], c['ymult'], c['yoff'], c['yzero'],
            c['scale'], c['position']) = \
            self.talk([
                'data:encdg rpbinary',
                f'data:source ch{i}',
                'curve?',
                'wfmpre:xincr?',
                'wfmpre:ymult?',
                'wfmpre:yoff?',
                'wfmpre:yzero?',
                f'ch{i}:scale?',
                f'ch{i}:position?'])
            c['offset'] = c['scale']*c['position']
            b_min = min(c['bytes'])
            b_max = max(c['bytes'])
            c['quality'] = 0 if b_min == 0 or b_max == 255 \
                else (b_max-b_min)/2
            c['u'] = [(b-c['yoff'])*c['ymult']+c['yzero']
                for b in c['bytes']]
            c['u_err'] = 0 # TODO
        return v
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
