#!/usr/bin/env python3
#
# -*- coding: utf-8 -*-
#
# Program dokonuje pomiaru napięcia generatora.  Wyjście 1 generatora jest
# połączone z wejściem kanału 1 oscyloskopu.  Program wykonuje serię
# pomiarów.  Zmienia częstotliwość sygnału w generatorze i rejestruje sygnał
# w oscyloskopie.
#
# Paweł Klimczewski, 6 listopada 2024

# Schemat pracy programu.
#
# 1. Przygotuj generator.
# 2. Przygotuj oscyloskop.
# 3. f = FMIN
# 4. Powtarzaj dopóki f<=FMAX.
#    4.1 Ustaw częstotliwość f w generatorze.
#    4.2 Ustaw jednostkę czasu na osi OX oscyloskopu równą 3/f/12.
#    4.3 Powtórz nie więcej niż MAXITER razy.
#        4.3.1 W spowolnionej komunikacji z oscyloskopem rozpocznij
#              pojedynczy pomiar i poczekaj na jego zakończenie.
#        4.3.2 Odczytaj wyniki pomiaru i dopisz je do listy wszystkich
#              wyników.
#        4.3.3 Sprawdź czy można dobrać bardziej korzystne wartości
#              jednostki napięcia i przesunięcia na osi OY. Jeżeli
#              nie, to przerwij pętlę. Jeżeli tak kontynuuj pętlę.
#    4.4 Oblicz kolejną wartość f = f*10^(1/N)
# 5. Odłącz sygnał w gnieździe generatora i wyłącz kanał pomiarowy
#    oscyloskopu.
# 6. Zapisz wyniki do pliku.

# Biblioteka tik zawiera trzy klasy RigolGenerator, RigolOscilloscope,
# TektronixOscilloscope umożliwiające sterowanie poszczególnymi
# urządzeniami.
from tik import RigolGenerator, RigolOscilloscope
import json

# Stałe określające parametry pracy programu.
#
# Częstotliwość początkowa [Hz].
FMIN = 1.e-1
# Częstotliwość końcowa [Hz].
FMAX = 2.e+7
# Liczba kroków na dekadę podczas zmiany częstotliwości. Kolejne
# częstotliwości są wyrazami ciągu geometrycznego o stałej q=10^(1/N).
N = 10
# Napięcie wyjściowe generatora [V].
VPP = 16
# Maksymalna liczba powtórzeń pomiaru dla pojedynczej częstotliwości
# sygnału.
MAXITER = 25
# Plik z wynikami pomiarów.
OUTFILENAME = 'generator.json'

# Lista, na której będą zapisywane wyniki pomiarów.
results = []
try:
    with RigolGenerator() as g, RigolOscilloscope() as o:

        # 1. Przygotowanie generatora.
        g.talk([
            'output:load infinity',
            'voltage:unit vpp',
            # Wartości liczbowe będące argumentami rozkazów przesyłanych do
            # przyrządów zaokrąglam do 3 cyfr znaczących.
            f'apply:sinusoid {FMIN:.2e},{VPP:.2e},0',
            'output on'])

        # 2. Przygotowanie oscyloskopu.
        o.talk([
            '*cls',
            '*rst',
            # Jako źródło impulsów synchronizacji wybieram napięcie sieci
            # energetycznej. Pomiar rozpocznie się najpóźniej 20 ms po
            # rozkazie single.
            'trigger:edge:source ac',
            'channel1:probe 1',
            f'channel1:scale {VPP/8:.2e}'])

        # 3.
        freq = FMIN
        # Porównując bieżącą częstotliwość z FMAX zaokrąglam ją do 3 cyfr
        # znaczących.
        # 4.
        while float(f'{freq:.2e}')<=FMAX:

            # 4.1 Ustawiam częstotliwość generatora.
            g.talk(f'frequency {freq:.2e}')

            # 4.2 Dostosowuję podstawę czasu w oscyloskopie.  Chcę aby na
            # ekranie widoczne były mniej więcej 3 okresy.  Oscyloskop
            # zaokrągli podaną wartość.  Stała o.NDIVX określa liczbę
            # jednostek na osi OX oscyloskopu.
            o.talk(f'timebase:main:scale {3./freq/o.NDIVX:.2e}')

            # 4.3 Wykonuję pojedynczy pomiar.  Być może pomiar będzie musiał
            # zostać powtórzony.  Dlatego korzystam z instrukcji
            # iteracyjnej.  Wprowadzam ograniczenie MAXITER pomiarów dla
            # jednej częstotliwości.
            for i in range(MAXITER):
                # 4.3.1 Wykonuję pojedynczy pomiar.  Oscyloskop Rigol jest
                # bardzo wrażliwy podczas wykonywania pomiaru.  Zbyt szybka
                # komunikacja z komputerem może powodować błędy w pracy.
                # Obiekt slower spowalnia komunikację w bloku with.
                with o.slower:
                    o.talk('single')
                    while o.talk('trigger:status?')!='STOP':
                        pass
                # 4.3.2 Pomiar jest zakończony.  Odczytuję dane
                # zarejestrowanych sygnałów.  W słowniku v zapisywane są
                # następujące wartości.
                # v['time'] - czas pomiaru
                # v['time_scale'] - jednostka na osi OX [s/div]
                # v['ch1']['scale'] - jednostka na osi OY [V/div]
                # v['ch1']['offset'] - przesunięcie w pionie [V]
                # v['ch1']['u'] - napięcia pomiarów [V]
                # v['ch1']['u_err'] - niepewności pomiarów [V]
                v = o.read_waveforms([1])
                # Częstotliwość sygnału [Hz]
                v['freq'] = freq
                # Napięcie międzyszczytowe obliczone przez oscyloskop [V]
                v['vpp1'] = o.talk('measure:item? vpp,channel1')

                results.append(v)

                # 4.3.3 Sprawdzam czy pomiar należy powtórzyć.  Jeżeli można
                # zaproponować bardziej korzystne wartości jednostki na osi
                # OY i przesunięcia wykresu w pionie, to zostaną one
                # ustawione w oscyloskopie a rezultatem funkcji będzie
                # wartość prawda.
                if not o.autoscale([1], v):
                    break

            # 4.4 Obliczam wartość kolejnej częstotliwości.
            freq = freq * (10.**(1./N))

        # 5. Końcowe porządki.
        g.talk('output off')
        o.talk('channel1:display off')
finally:
    # 6. Wyniki zapisuję to pliku.
    with open(OUTFILENAME, 'w') as f:
        f.write(json.dumps(results))
